# Copyright 2014 Tom Rothamel <tom@rothamel.us>
# Copyright 2014 Patrick Dawson <pat@dw.is>
#
# This software is provided 'as-is', without any express or implied
# warranty.  In no event will the authors be held liable for any damages
# arising from the use of this software.
#
# Permission is granted to anyone to use this software for any purpose,
# including commercial applications, and to alter it and redistribute it
# freely, subject to the following restrictions:
#
# 1. The origin of this software must not be misrepresented; you must not
#    claim that you wrote the original software. If you use this software
#    in a product, an acknowledgment in the product documentation would be
#    appreciated but is not required.
# 2. Altered source versions must be plainly marked as such, and must not be
#    misrepresented as being the original software.
# 3. This notice may not be removed or altered from any source distribution.

import sys
from typing import Callable


# Lists of functions that are called on init and quit.
init_functions: list[Callable[[], None]] = []
quit_functions: list[Callable[[], None]] = []

def register_init(fn):
    init_functions.append(fn)
    return fn

def register_quit(fn):
    quit_functions.append(fn)
    return fn

def init():
    numpass = 0
    numfail = 0

    for i in init_functions:
        try:
            i()
            numpass += 1
        except Exception:
            numfail += 1

    return numpass, numfail

def quit():
    for i in quit_functions:
        try:
            i()
        except Exception:
            pass


# Import core modules.
from renpy.pygame.error import *

from renpy.pygame.surface import Surface as Surface
from renpy.pygame.rect import Rect as Rect

from renpy.pygame import (
    rect as rect,
    surface as surface,
    color as color,
    display as display,
    event as event,
    key as key,
    locals as locals,
    time as time,
    version as version,
    locals as constants,
    controller as controller,
    draw as draw,
    image as image,
    joystick as joystick,
    mouse as mouse,
    power as power,
    transform as transform,
    scrap as scrap,
    sysfont as sysfont,
)


# Fill this module with locals.
from renpy.pygame.locals import *


def import_as_pygame():
    """
    Imports pygame_sdl2 as pygame, so that running the 'import pygame.whatever'
    statement will import renpy.pygame.whatever instead.
    """

    import warnings

    if "pygame" in sys.modules:
        warnings.warn("Pygame has already been imported, import_as_pygame may not work.", stacklevel=2)

    if "pygame_sdl2" in sys.modules:
        warnings.warn("Pygame SDL2 has already been imported, import_as_pygame may not work.", stacklevel=2)

    for name, mod in list(sys.modules.items()):
        if name.startswith("renpy.pygame"):
            suffix = name[len("renpy.pygame") :]
        else:
            continue

        sys.modules["pygame" + suffix] = mod
        sys.modules["pygame_sdl2" + suffix] = mod

    sys.modules["pygame.constants"] = constants
    sys.modules["pygame_sdl2.constants"] = constants


def get_sdl_byteorder():
    return BYTEORDER


def get_sdl_version():
    return SDL_VERSION_TUPLE


def get_platform() -> str:
    return display.get_platform()


# We have new-style buffers, but not the pygame.newbuffer module.
HAVE_NEWBUF = False


# Generated by scripts/relative_imports.py, do not edit below this line.
import typing

if typing.TYPE_CHECKING:
    from . import color as color
    from . import controller as controller
    from . import display as display
    from . import draw as draw
    from . import error as error
    from . import event as event
    from . import gfxdraw as gfxdraw
    from . import image as image
    from . import joystick as joystick
    from . import key as key
    from . import locals as locals
    from . import mouse as mouse
    from . import power as power
    from . import pygame_time as pygame_time
    from . import rect as rect
    from . import rwobject as rwobject
    from . import scrap as scrap
    from . import surface as surface
    from . import sysfont as sysfont
    from . import time as time
    from . import transform as transform
    from . import version as version
